//===============================================================================
// Microsoft Exception Management Application Block for .NET
// http://msdn.microsoft.com/library/en-us/dnbda/html/emab-rm.asp
//
// ExceptionManager.cs
// This file contains the ExceptionManager class, which manages all publishing 
// of exceptions, and the default publisher class, which publishes exception 
// information to the Event Log.
//
// For more information see the Implementing the Exception Manager Component
// section of the Exception Management Application Block Implementation Overview. 
//===============================================================================
// Copyright (C) 2000-2001 Microsoft Corporation
// All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Resources;
using System.Reflection;
using System.Collections;
using System.Configuration;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using System.Runtime.Serialization;
using System.Threading;
using System.Collections.Specialized;
using System.Security;
using System.Security.Principal;
using System.Security.Permissions;
using gov.va.med.vbecs.Common.Log;

namespace gov.va.med.vbecs.ExceptionManagement
{
    #region Header

    ///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    ///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    ///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    ///<Developers>
    ///	<Developer>Carrie Van Stedum</Developer>
    ///</Developers>
    ///<SiteName>Hines OIFO</SiteName>
    ///<CreationDate>03/12/2003</CreationDate>
    ///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    /// <summary>
    /// The Exception Manager class manages the publishing of exception information based on settings in the configuration file.
    /// </summary>
    #endregion

    #region ExceptionManager Class

    public class ExceptionManager
    {
        /// <summary>
        /// Private constructor to restrict an instance of this class from being created.
        /// </summary>
        private ExceptionManager()
        {
        }

        // Member variable declarations
        private readonly static string EXCEPTIONMANAGER_NAME = typeof(ExceptionManager).Name;

        private static ResourceManager resourceManager = new ResourceManager("gov.va.med.vbecs.Common.ExceptionManagerText", Assembly.GetAssembly(typeof(ExceptionManager)));
        private const string TEXT_SEPARATOR = "*********************************************";

        ///<Developers>
        ///<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>6/10/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3087"> 
        ///		<ExpectedInput>Exception</ExpectedInput>
        ///		<ExpectedOutput>Raised Exception</ExpectedOutput>
        ///</Case>
        ///
        ///<Case type="1" testid ="3089"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// Static method to publish the exception information.
        /// </summary>
        /// <param name="exception">The exception object whose information should be published.</param>
        public static void Publish(Exception exception)
        {
            ExceptionManager.Publish(exception, null);
        }

        ///<Developers>
        ///<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>6/10/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3093"> 
        ///		<ExpectedInput>Exception</ExpectedInput>
        ///		<ExpectedOutput>Raised Exception</ExpectedOutput>
        ///</Case>
        ///
        ///<Case type="1" testid ="3095"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// This method is used to publish exception information.  Additional information can be published along with 
        /// the exception information.
        /// </summary>
        /// <param name="exception">The exception object whose information should be published.</param>
        /// <param name="additionalInfo">A collection of additional data that should be published along with the exception information.</param>
        public static void Publish(Exception exception, NameValueCollection additionalInfo)
        {
            #region Load the AdditionalInformation Collection with environment data.
            // Create the Additional Information collection if it does not exist.
            if (null == additionalInfo)
            {
                additionalInfo = new NameValueCollection();
            }

            // Add environment information to the information collection.
            try
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".MachineName", Environment.MachineName);
            }
            catch (SecurityException)
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".MachineName", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED"));
            }
            catch
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".MachineName", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION"));
            }

            try
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".TimeStamp", DateTime.Now.ToString());
            }
            catch (SecurityException)
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".TimeStamp", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED"));
            }
            catch
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".TimeStamp", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION"));
            }

            try
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".FullName", Assembly.GetExecutingAssembly().FullName);
            }
            catch (SecurityException)
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".FullName", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED"));
            }
            catch
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".FullName", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION"));
            }

            try
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".AppDomainName", AppDomain.CurrentDomain.FriendlyName);
            }
            catch (SecurityException)
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".AppDomainName", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED"));
            }
            catch
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".AppDomainName", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION"));
            }

            try
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".ThreadIdentity", Thread.CurrentPrincipal.Identity.Name);
            }
            catch (SecurityException)
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".ThreadIdentity", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED"));
            }
            catch
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".ThreadIdentity", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION"));
            }

            try
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".WindowsIdentity", WindowsIdentity.GetCurrent().Name);
            }
            catch (SecurityException)
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".WindowsIdentity", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED"));
            }
            catch
            {
                additionalInfo.Add(EXCEPTIONMANAGER_NAME + ".WindowsIdentity", resourceManager.GetString("RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION"));
            }

            #endregion

            Publish(exception, additionalInfo, null);

        } // End Publish(Exception exception, NameValueCollection AdditionalInfo)

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/29/2005</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="8120"> 
        ///		<ExpectedInput>exceptionObject</ExpectedInput>
        ///		<ExpectedOutput>Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///<Case type="1" testid ="8121"> 
        ///		<ExpectedInput>null</ExpectedInput>
        ///		<ExpectedOutput>BaseApplicationException with Unknown text</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Converts supplied exception object which can be a native or managed exception 
        /// or null to a managed exception object. If null is supplied the method will 
        /// provide an artificial exception explaining the issue. 
        /// </summary>
        /// <param name="exceptionObject">Exception object, can be native/managed exception or null.</param>
        /// <returns>Managed exception.</returns>
        /// <remarks>
        ///		PLEASE DO NOT MOVE STRINGS IN THIS METHOD TO SATELLITE ASSEMBLY. 
        ///		This method must work even if resource assembly is not found or string resource manager fails. 
        ///	</remarks>
        public static Exception ConvertExceptionObjectToManagedException(object exceptionObject)
        {
            if (exceptionObject == null)
                return new BaseApplicationException("Unknown error occurred - exception object is null.");

            Exception _managedException = exceptionObject as Exception;

            if (_managedException != null)
                return _managedException;

            return new BaseApplicationException(String.Format("Native Windows exception of type {0} occurred. Full exception object text: {1}.", exceptionObject.GetType().Name, exceptionObject.ToString()));
        }

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/24/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3109"> 
        ///		<ExpectedInput>Exception, NameValueCollection, NameValueCollection</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///	
        ///<Case type="1" testid ="3110"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// This method is used to publish exception information.  Additional information can be published along with 
        /// the exception information.
        /// </summary>
        /// <param name="exception">The exception object whose information should be published.</param>
        /// <param name="additionalInfo">A collection of additional data that should be published along with the exception information.</param>
        /// <param name="configSettings">A collection of any additional attributes provided in the config settings for the custom publisher.</param>
        public static void Publish(Exception exception, NameValueCollection additionalInfo, NameValueCollection configSettings)
        {
            // Create StringBuilder to maintain publishing information.
            var strInfo = new StringBuilder();

            #region Record the contents of the AdditionalInfo collection
            // Record the contents of the AdditionalInfo collection.
            if (additionalInfo != null)
            {
                // Record General information.
                strInfo.AppendFormat("{0}General Information {0}{1}{0}Additional Info:", Environment.NewLine, TEXT_SEPARATOR);

                foreach (string i in additionalInfo)
                {
                    strInfo.AppendFormat("{0}{1}: {2}", Environment.NewLine, i, additionalInfo.Get(i));
                }
            }
            #endregion

            if (exception == null)
            {
                strInfo.AppendFormat("{0}{0}No Exception object has been provided.{0}", Environment.NewLine);
            }
            else
            {
                strInfo.Append(FormatExceptionInfoAndDumpToString(exception));
            }

            // Write the entry to the event log. 
            // EventLogAppender
            var eventsLogger = LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");
            eventsLogger.Error(strInfo.ToString());

            //Write the entry to error log file
            if (Common.VbecsConfig.Current.ErrorLogFileName.Length > 0)
            {

                string formattedException = ExceptionManagement.ExceptionManager.FormatExceptionInfoAndDumpToString(exception);

                using (FileStream fs = File.Open(Common.VbecsConfig.Current.ErrorLogFileName, FileMode.Append, FileAccess.Write))
                {
                    using (StreamWriter sw = new StreamWriter(fs))
                    {
                        sw.Write(formattedException);
                    }
                }
            }
        }


        ///<Developers>
        ///	<Developer>Stas Antropov</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>7/16/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3050"> 
        ///		<ExpectedInput>Valid exception</ExpectedInput>
        ///		<ExpectedOutput>string</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="3051"> 
        ///		<ExpectedInput>null</ExpectedInput>
        ///		<ExpectedOutput>String</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Puts all exception information into a string
        /// </summary>
        /// <param name="xcp"></param>
        /// <returns></returns>
        public static string FormatExceptionInfoAndDumpToString(Exception xcp)
        {
            if (xcp == null)
                return "No exception information available";

            System.Text.StringBuilder strInfo = new System.Text.StringBuilder();

            strInfo.Append(Environment.NewLine);
            strInfo.Append(Environment.NewLine);
            strInfo.Append(GetTextSeparatorWithNewlines());
            strInfo.Append(GetExceptionHeader());
            strInfo.Append(GetTextSeparatorWithNewlines());
            strInfo.Append(GetEnvironmentInfo());
            strInfo.Append(GetTextSeparatorWithNewlines());

            // Loop through each exception class in the chain of exception objects.
            Exception currentException = xcp;	// Temp variable to hold InnerException object during the loop.
            int intExceptionCount = 1;				// Count variable to track the number of exceptions in the chain.
            do
            {
                // Write title information for the exception object.
                strInfo.AppendFormat("{0}{0}{1}) Exception Information{0}{2}", Environment.NewLine, intExceptionCount.ToString(), TEXT_SEPARATOR);
                strInfo.AppendFormat("{0}Exception Type: {1}", Environment.NewLine, currentException.GetType().FullName);

                #region Loop through the public properties of the exception object and record their value
                // Loop through the public properties of the exception object and record their value.
                PropertyInfo[] aryPublicProperties = currentException.GetType().GetProperties();
                NameValueCollection currentAdditionalInfo;
                foreach (PropertyInfo p in aryPublicProperties)
                {
                    // Do not log information for the InnerException or StackTrace. This information is 
                    // captured later in the process.
                    if (p.Name != "InnerException" && p.Name != "StackTrace")
                    {
                        if (p.GetValue(currentException, null) == null)
                        {
                            strInfo.AppendFormat("{0}{1}: NULL", Environment.NewLine, p.Name);
                        }
                        else
                        {
                            // Loop through the collection of AdditionalInformation if the exception type is a BaseApplicationException.
                            if (p.Name == "AdditionalInformation" && currentException is BaseApplicationException)
                            {
                                // Verify the collection is not null.
                                if (p.GetValue(currentException, null) != null)
                                {
                                    // Cast the collection into a local variable.
                                    currentAdditionalInfo = (NameValueCollection)p.GetValue(currentException, null);

                                    // Check if the collection contains values.
                                    if (currentAdditionalInfo.Count > 0)
                                    {
                                        strInfo.AppendFormat("{0}AdditionalInformation:", Environment.NewLine);

                                        // Loop through the collection adding the information to the string builder.
                                        for (int i = 0; i < currentAdditionalInfo.Count; i++)
                                        {
                                            strInfo.AppendFormat("{0}{1}: {2}", Environment.NewLine, currentAdditionalInfo.GetKey(i), currentAdditionalInfo[i]);
                                        }
                                    }
                                }
                            }
                            // Otherwise just write the ToString() value of the property.
                            else
                            {
                                strInfo.AppendFormat("{0}{1}: {2}", Environment.NewLine, p.Name, p.GetValue(currentException, null));
                            }
                        }
                    }
                }
                #endregion
                #region Record the Exception StackTrace
                // Record the StackTrace with separate label.
                if (currentException.StackTrace != null)
                {
                    strInfo.AppendFormat("{0}{0}StackTrace Information{0}{1}", Environment.NewLine, TEXT_SEPARATOR);
                    strInfo.AppendFormat("{0}{1}", Environment.NewLine, currentException.StackTrace);
                }
                #endregion

                // Reset the temp exception object and iterate the counter.
                currentException = currentException.InnerException;
                intExceptionCount++;
            } while (currentException != null);


            return strInfo.ToString();
        }
        private static string GetTextSeparatorWithNewlines()
        {
            return String.Concat(Environment.NewLine, TEXT_SEPARATOR, Environment.NewLine);
        }

        private static string GetExceptionHeader()
        {
            return "VBECS - Unexpected Exception";
        }

        private static string GetEnvironmentInfo()
        {
            StringBuilder _sb = new StringBuilder();

            DateTime _currentDt = DateTime.Now;

            _sb.AppendFormat("Local Time: {0} {1}{2}", _currentDt.ToShortDateString(), _currentDt.ToShortTimeString(), Environment.NewLine);
            _sb.AppendFormat("Machine Name: {0}{1}", Environment.MachineName, Environment.NewLine);
            _sb.AppendFormat("NT User: {0}\\{1}{2}", Environment.UserDomainName, Environment.UserName, Environment.NewLine);
            _sb.AppendFormat("Current Folder: {0}{1}", Environment.CurrentDirectory, Environment.NewLine);
            _sb.AppendFormat("OS Version: {0} {1}", Environment.OSVersion.ToString(), Environment.NewLine);
            _sb.AppendFormat("CLR Version: {0} {1}", Environment.Version.ToString(), Environment.NewLine);

            return _sb.ToString();
        }
    }
    #endregion
}
    /*
    #region DefaultPublisher Class
	/// <summary>
	/// Component used as the default publishing component if one is not specified in the config file.
	/// 
	/// Some code was fixed for DefaultPublisher (CR2761). 
	/// Now it is simplified and uses thread safe EventsLoggerSingleton to write events.
	/// </summary>
	public sealed class DefaultPublisher : IExceptionPublisher
	{
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3107"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DefaultPublisher</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3108"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Default Constructor.
		/// </summary>
		public DefaultPublisher()
		{
		}

		private static ResourceManager resourceManager = new ResourceManager("gov.va.med.vbecs.Common.ExceptionManagerText",Assembly.GetAssembly(typeof(ExceptionManager)));
		private const string TEXT_SEPARATOR = "*********************************************";
		

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3109"> 
		///		<ExpectedInput>Exception, NameValueCollection, NameValueCollection</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3110"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// This method is used to publish exception information.  Additional information can be published along with 
		/// the exception information.
		/// </summary>
		/// <param name="exception">The exception object whose information should be published.</param>
		/// <param name="additionalInfo">A collection of additional data that should be published along with the exception information.</param>
		/// <param name="configSettings">A collection of any additional attributes provided in the config settings for the custom publisher.</param>
		public void Publish(Exception exception, NameValueCollection additionalInfo, NameValueCollection configSettings)
		{
			// Create StringBuilder to maintain publishing information.
			StringBuilder strInfo = new StringBuilder();

			#region Record the contents of the AdditionalInfo collection
			// Record the contents of the AdditionalInfo collection.
			if (additionalInfo != null)
			{
				// Record General information.
				strInfo.AppendFormat("{0}General Information {0}{1}{0}Additional Info:", Environment.NewLine, TEXT_SEPARATOR);

				foreach (string i in additionalInfo)
				{
					strInfo.AppendFormat("{0}{1}: {2}", Environment.NewLine, i, additionalInfo.Get(i));
				}
			}
			#endregion

			if (exception == null)
			{
				strInfo.AppendFormat("{0}{0}No Exception object has been provided.{0}", Environment.NewLine);
			}
			else
			{
				strInfo.Append( FormatExceptionInfoAndDumpToString( exception ) );
			}

			// Write the entry to the event log.   
      		Common.EventsLoggerSingleton.Instance.Log(strInfo.ToString(), EventLogEntryType.Error);
		}


		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3050"> 
		///		<ExpectedInput>Valid exception</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3051"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>String</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Puts all exception information into a string
		/// </summary>
		/// <param name="xcp"></param>
		/// <returns></returns>
		public static string FormatExceptionInfoAndDumpToString( Exception xcp )
		{
			if( xcp == null )
				return "No exception information available";

			System.Text.StringBuilder strInfo = new System.Text.StringBuilder();

			strInfo.Append( Environment.NewLine );
			strInfo.Append( Environment.NewLine );
			strInfo.Append( GetTextSeparatorWithNewlines() );
			strInfo.Append( GetExceptionHeader() );
			strInfo.Append( GetTextSeparatorWithNewlines() );
			strInfo.Append( GetEnvironmentInfo() );
			strInfo.Append( GetTextSeparatorWithNewlines() );

			// Loop through each exception class in the chain of exception objects.
			Exception currentException = xcp;	// Temp variable to hold InnerException object during the loop.
			int intExceptionCount = 1;				// Count variable to track the number of exceptions in the chain.
			do
			{				
				// Write title information for the exception object.
				strInfo.AppendFormat("{0}{0}{1}) Exception Information{0}{2}", Environment.NewLine, intExceptionCount.ToString(), TEXT_SEPARATOR);
				strInfo.AppendFormat("{0}Exception Type: {1}", Environment.NewLine, currentException.GetType().FullName);
				
				#region Loop through the public properties of the exception object and record their value
				// Loop through the public properties of the exception object and record their value.
				PropertyInfo[] aryPublicProperties = currentException.GetType().GetProperties();
				NameValueCollection currentAdditionalInfo;
				foreach (PropertyInfo p in aryPublicProperties)
				{
					// Do not log information for the InnerException or StackTrace. This information is 
					// captured later in the process.
					if (p.Name != "InnerException" && p.Name != "StackTrace")
					{
						if (p.GetValue(currentException,null) == null)
						{
							strInfo.AppendFormat("{0}{1}: NULL", Environment.NewLine, p.Name);
						}
						else
						{
							// Loop through the collection of AdditionalInformation if the exception type is a BaseApplicationException.
							if (p.Name == "AdditionalInformation" && currentException is BaseApplicationException)
							{
								// Verify the collection is not null.
								if (p.GetValue(currentException,null) != null)
								{
									// Cast the collection into a local variable.
									currentAdditionalInfo = (NameValueCollection)p.GetValue(currentException,null);

									// Check if the collection contains values.
									if (currentAdditionalInfo.Count > 0)
									{
										strInfo.AppendFormat("{0}AdditionalInformation:", Environment.NewLine);

										// Loop through the collection adding the information to the string builder.
										for (int i = 0; i < currentAdditionalInfo.Count; i++)
										{
											strInfo.AppendFormat("{0}{1}: {2}", Environment.NewLine, currentAdditionalInfo.GetKey(i), currentAdditionalInfo[i]);
										}
									}
								}
							}
								// Otherwise just write the ToString() value of the property.
							else
							{
								strInfo.AppendFormat("{0}{1}: {2}", Environment.NewLine, p.Name, p.GetValue(currentException,null));
							}
						}
					}
				}
					#endregion
				#region Record the Exception StackTrace
				// Record the StackTrace with separate label.
				if (currentException.StackTrace != null)
				{
					strInfo.AppendFormat("{0}{0}StackTrace Information{0}{1}", Environment.NewLine, TEXT_SEPARATOR);
					strInfo.AppendFormat("{0}{1}", Environment.NewLine, currentException.StackTrace);
				}
				#endregion

				// Reset the temp exception object and iterate the counter.
				currentException = currentException.InnerException;
				intExceptionCount++;
			} while (currentException != null);


			return strInfo.ToString();
		}

		private static string GetTextSeparatorWithNewlines()
		{
			return String.Concat( Environment.NewLine, TEXT_SEPARATOR, Environment.NewLine );
		}

		private static string GetExceptionHeader()
		{
			return "VBECS - Unexpected Exception";
		}

		private static string GetEnvironmentInfo()
		{
			StringBuilder _sb = new StringBuilder();
			
			DateTime _currentDt = DateTime.Now;

			_sb.AppendFormat( "Local Time: {0} {1}{2}", _currentDt.ToShortDateString(), _currentDt.ToShortTimeString(), Environment.NewLine );
			_sb.AppendFormat( "Machine Name: {0}{1}", Environment.MachineName, Environment.NewLine );
			_sb.AppendFormat( "NT User: {0}\\{1}{2}", Environment.UserDomainName, Environment.UserName, Environment.NewLine );			
			_sb.AppendFormat( "Current Folder: {0}{1}", Environment.CurrentDirectory, Environment.NewLine );
			_sb.AppendFormat( "OS Version: {0} {1}", Environment.OSVersion.ToString(), Environment.NewLine );
			_sb.AppendFormat( "CLR Version: {0} {1}", Environment.Version.ToString(), Environment.NewLine );

			return _sb.ToString();
		}
	}
	#endregion*/
